/* PC parallel port DAC speaker driver. Accepts 8 bit unsigned 16k mono audio data */
#include <OS.h>
#include <SupportDefs.h>
#include <KernelExport.h>
#include <Drivers.h>
#include <math.h>
#include <ISA.h>
#include "audio_sound_driver.h"

/*****************************************************************************************************************
 This defines the maximum interrupt latency in microseconds.  We generally always want to interrupt early and spin
 until the proper time.  In general, the actual overhead for an interrupt is small, however, if other drivers have
 interrupts disabled or are servicing interrupts, the actual latency could increase. */

#define DEFAULT_SAMPLE_RATE        16000.0
#define DEFAULT_CHANNELS           2
#define DEFAULT_FORMAT             0x12 // (uchar).
#define DEFAULT_ENDIAN             0    // 0 for little endian, 1 for big endian.
#define DEFAULT_BUFFER_HEADER      0    // ??????
#define DEFAULT_WRITE_BUFFER_SIZE  B_PAGE_SIZE
#define DEFAULT_RECORD_BUFFER_SIZE B_PAGE_SIZE // 0 Should tell OS no recording available?

/* Used to load and unload driver */
status_t init_hardware (void);
status_t init_driver(void);
void uninit_driver(void);

/* Communication between BeOS & Driver */
static status_t device_open(const char *name, uint32 flags, void **cookie);
static status_t device_close(void *cookie);
static status_t device_free(void *cookie);
static status_t device_control(void *cookie, uint32 op, void *buf, size_t length);
static status_t device_read(void *cookie, off_t pos, void *buf, size_t *length);
static status_t device_write(void *cookie, off_t pos, const void *buf, size_t *length);

static status_t set_parameters(sound_setup *setup);
static status_t get_parameters(sound_setup *setup);

/* Internal Data */
const char **publish_devices(void);
device_hooks* find_device(const char *name);

// Note:  The following variables would normally be declared as locals or allocated from memory
// but due to bugs in coding I am placing here to help debug the code.
static audio_format AudioFormat;  // Declared to help debug code.  Should be a mallocated array.

static uint16 play_buffer[32768];   // Buffer 16 bit stereo data of sound to be played.
static uint16 record_buffer[32768]; // Buffer 16 bit stereo data of sound presently being recorded.
static uint8  sound_data8[32768];   // Buffer of 8 bit mono sound presently being played.

static bigtime_t next_buffer_time;  // Next time slot available to start playing a sound buffer.
static long Latency_Timing = 0;     // Interrupt latency of output timing.

static int32 open_count = 0; // Number of times driver openned.
static isa_module_info *isa = NULL;
static sem_id write_done_sem = -1, read_done_sem = -1, write_block = -1, read_block = -1;
static bool info_write = true, info_read = true;

static char left_adc_source = line, right_adc_source = line, // Storage varibles for mixer controls.
            left_adc_gain  = 15, right_adc_gain = 15,

            left_mic_gain_enable = true, right_mic_gain_enable = true,

            left_aux1_mix_gain  = 31, left_aux1_mix_mute  = false,
            right_aux1_mix_gain = 31, right_aux1_mix_mute = false,

            left_aux2_mix_gain  = 31, left_aux2_mix_mute  = false,
            right_aux2_mix_gain = 31, right_aux2_mix_mute = false,

            left_line_mix_gain  = 31, left_line_mix_mute  = false,
            right_line_mix_gain = 31, right_line_mix_mute = false,

            left_dac_attn  = 31, left_dac_mute  = false,
            right_dac_attn = 31, right_dac_mute = false,

            sound_sample_rate = kHz_16_0,
            sound_playback_format = linear_16bit_little_endian_stereo, /* Obsolete? Always signed 16bit-linear. */
            sound_capture_format  = linear_16bit_little_endian_stereo, /* Obsolete? Always signed 16bit-linear. */

            sound_dither_enable = true,
            sound_loop_attn = 31, sound_loop_enable = true,
            sound_output_boost = true, sound_highpass_enable = true,
            sound_mono_gain = 31, sound_mono_mute = false ;

/*****************************************************************************************************************
 api_version - This variable defines the API version to which the driver was written, and should be set to
 B_CUR_DRIVER_API_VERSION at compile time. The value of this variable will be changed with every revision to the
 driver API; the value with which you compiled with will tell devfs how it can communicate with the driver. */

int32 api_version = B_CUR_DRIVER_API_VERSION;

const char *driver_names[] = { "audio/old/sound", NULL };

/*****************************************************************************************************************
 Device Hooks - The hook functions specified in the device_hooks function returned by the driver's find_device()
 function handle requests made by devfs (and through devfs, from user applications). These are described in this
 section.  The structure itself looks like this: 
 typedef struct { 
  device_open_hook open;       -> open entry point
  device_close_hook close;     -> close entry point
  device_free_hook free;       -> free cookie
  device_control_hook control; -> control entry point
  device_read_hook read;       -> read entry point
  device_write_hook write;     -> write entry point
  device_select_hook select;   -> select entry point
  device_deselect_hook deselect; -> deselect entry point
  device_readv_hook readv;     -> posix read entry point
  device_writev_hook writev;   -> posix write entry point
 } device_hooks;
 In all cases, return B_OK if the operation is successfully completed, or an appropriate error code if not.
 function pointers for the device hooks entry points */

static device_hooks sound_device[] = { // NULLs=sound_select, sound_deselect, device_readv, device_writev
 device_open, device_close, device_free, device_control, device_read, device_write, NULL, NULL, NULL, NULL };

/*****************************************************************************************************************
 init_hardware - status_t init_hardware (void) - This function is called when the system is booted, which lets the
 driver detect and reset the hardware it controls. The function should return B_OK if the initialization is
 successful; otherwise, an appropriate error code should be returned. If this function returns an error, the
 driver won't be used. */

status_t init_hardware (void) {
int index = 0; uint16 count = 0;

 // dprintf("PPA: Initialize Hardware.\n\n"); 
 next_buffer_time = system_time(); // Setup for when first buffer of sound can be played.
 for (index = 0; index < 32767; index++ ) {
     record_buffer[index] = count; count = count + 32; } // Fill sound buffers.
 return B_OK; } 

/*****************************************************************************************************************
 init_driver - status_t init_driver(void) - optional function - called every time the driver is loaded.  Drivers
 are loaded and unloaded on an as-needed basis. When a driver is loaded by devfs, this function is called to let
 the driver allocate memory and other needed system resources. Return B_OK if initialization succeeds, otherwise
 return an appropriate error code. <<<what happens if this returns an error?>>> */

status_t init_driver(void) {
 status_t ret = B_OK; // dprintf("PPA: Initializing Driver.\n");

 ret = get_module(B_ISA_MODULE_NAME, (module_info**) &isa);
 if (ret == B_OK) {
     // dprintf("PPA: Get B_ISA_MODULE_NAME ok.\n"); 

    write_done_sem = create_sem(0, "write finished"); ret = write_done_sem;
    if (write_done_sem > B_OK) {
        // dprintf("PPA: Created write_done_sem ok.\n");

       write_block = create_sem(1, "write lock"); ret = write_block;
       if (write_block > B_OK) {
           // dprintf("PPA: Created write_lock sem ok.\n");

          read_done_sem = create_sem(0, "read finished"); ret = write_done_sem;
          if (read_done_sem > B_OK) {
              // dprintf("PPA: Created read_done_sem ok.\n");

             read_block  = create_sem(1, "read  lock"); ret = read_block;
             if (read_block > B_OK) {
                // dprintf("PPA: Created read_lock sem ok.\n");
                return B_OK; }

             delete_sem(read_done_sem); // dprintf("PPA: Deleted read_done_sem.\n");
             } 
          delete_sem(write_block); // dprintf("PPA: Deleted write_lock.\n");
          }
       delete_sem(write_done_sem); // dprintf("PPA: Deleted write_done_sem.\n");
       } 
    put_module(B_ISA_MODULE_NAME); // dprintf("PPA: Deleted B_ISA_MODULE_NAME.\n");
    }

 // dprintf("\n"); // Spacer in debugger output.
 return ret; }

/*****************************************************************************************************************
 uninit_driver - void uninit_driver(void) - optional function - called every time the driver is unloaded.  This
 function is called by devfs just before the driver is unloaded from memory. This lets the driver clean up after
 itself, freeing any resources it allocated. */

void uninit_driver(void) {
 // dprintf("PPA: Uninitializing Driver.\n\n");
 return; }

/*****************************************************************************************************************
 open_hook() - status_t open_hook(const char *name, uint32 flags, void **cookie) - This hook function is called
 when a program opens one of the devices supported by the driver. The name of the device (as returned by
 publish_devices()) is passed in name, along with the flags passed to the Posix open() function. Cookie points to
 space large enough for you to store a single pointer. You can use this to store state information specific to the
 open() instance. If you need to track information on a per-open() basis, allocate the memory you need and store a
 pointer to it in **cookie. */

static status_t device_open(const char *name, uint32 flags, void **cookie) {
 if (atomic_add(&open_count, 1) < 1) { // Confirmed first time openned.
    AudioFormat.sample_rate        = DEFAULT_SAMPLE_RATE;
    AudioFormat.channels           = DEFAULT_CHANNELS;
    AudioFormat.format             = DEFAULT_FORMAT;
    AudioFormat.big_endian         = DEFAULT_ENDIAN;
    AudioFormat.buffer_header      = DEFAULT_BUFFER_HEADER;
    AudioFormat.write_buffer_size  = DEFAULT_WRITE_BUFFER_SIZE;
    AudioFormat.record_buffer_size = DEFAULT_RECORD_BUFFER_SIZE;
    // dprintf("PPA: OLD AudioFormat Address = %lx \n",&AudioFormat);
    // dprintf("New sample_rate = %d \n",(long)AudioFormat.sample_rate);
    // dprintf("New channels = %d \n",AudioFormat.channels);
    // dprintf("New format = %d \n",AudioFormat.format);
    // dprintf("New big_endian = %d \n",AudioFormat.big_endian);
    // dprintf("New buffer_header = %d \n",AudioFormat.buffer_header);
    // dprintf("New write_buffer_size = %d \n",AudioFormat.write_buffer_size);
    // dprintf("New record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
    return B_OK;
 } else { // Once open keeps count at 1.
    // dprintf("PPA: Driver opened already.\n\n");
    atomic_add(&open_count, -1); return B_ERROR; } }

/*****************************************************************************************************************
 close_hook() - status_t close_hook(void *cookie) - This hook is called when an open instance of the driver is
 closed using the close() Posix function. Note that because of the multithreaded nature of the BeOS, it's possible
 there may still be transactions pending, and you may receive more calls on the device. For that reason, you
 shouldn't free instance-wide system resources here. Instead, you should do this in free_hook(). However, if there
 are any blocked transactions pending, you should unblock them here. */

static status_t device_close(void *cookie) {
 // dprintf("PPA: Closed the Driver.\n\n");
 atomic_add(&open_count, -1); return B_OK; }

/*****************************************************************************************************************
 free_hook() - status_t free_hook(void *cookie) - This hook is called once all pending transactions on an open
 (but closing) instance of your driver are completed. This is where your driver should release instance wide
 system resources. free_hook() doesn't correspond to any Posix function. */

static status_t device_free(void *cookie) {
 // dprintf("PPA: Freed Driver.\n");
 delete_sem(read_block);  delete_sem(read_done_sem);  // dprintf("PPA: Deleted read semaphores.\n"); 
 delete_sem(write_block); delete_sem(write_done_sem); // dprintf("PPA: Deleted write semaphores.\n"); 
 put_module(B_ISA_MODULE_NAME);                       // dprintf("PPA: Deleted B_ISA_MODULE_NAME.\n\n");
 return B_OK; }

/*****************************************************************************************************************
 control_hook() - status_t control_hook(void *cookie, uint32 op, void *data, size_t length) - This hook handles
 the ioctl() function for an open instance of your driver. The control hook provides a means to perform operations
 that don't map directly to either read() or write(). It receives the cookie for the open instance, plus the
 command code op and the data and len arguments specified by ioctl()'s caller.  These arguments have no inherent
 relationship; they're simply arguments to ioctl() that are forwarded to your hook function. Their definitions are
 defined by the driver. Common command codes can be found in be/drivers/Drivers.h.  The len argument is only valid
 when ioctl() is called from user space; the kernel always sets it to 0. */

static status_t device_control(void *cookie, uint32 opcode, void *data_buffer, size_t length) {
 status_t err; audio_buffer_header *header; size_t data_length;
 static float rates[1] = {DEFAULT_SAMPLE_RATE }; // Works for me as far as testing goes.
// static float rates[5] = { 32000.0, 22050.0, 16000.0, 11025.0, 8000.0 };

 switch(opcode) {

    case SOUND_GET_PARAMS: // Copies parameters into a sound struct pointed to inside the data buffer.
         // dprintf("PPA: GET PARAMETERS\n");
         return get_parameters((sound_setup *)data_buffer);

    case SOUND_SET_PARAMS: // Copies parameters from a sound struct pointed to inside the data buffer.
         // dprintf("PPA: SET PARAMETERS\n");
         return set_parameters((sound_setup *)data_buffer);

    case B_AUDIO_SET_AUDIO_FORMAT: // arg = ptr to struct audio_format
         // dprintf("PPA: SET AUDIO FORMAT\n");
         // dprintf("Old sample_rate = %d \n",(long)AudioFormat.sample_rate);
         // dprintf("Old channels = %d \n",AudioFormat.channels);
         // dprintf("Old format = %d \n",AudioFormat.format);
         // dprintf("Old big_endian = %d \n",AudioFormat.big_endian);
         // dprintf("Old buffer_header = %d \n",AudioFormat.buffer_header);
         // dprintf("Old write_buffer_size = %d \n",AudioFormat.write_buffer_size);
         // dprintf("Old record_buffer_size = %d \n",AudioFormat.record_buffer_size);
         memcpy(&AudioFormat, data_buffer, min(sizeof(audio_format), length));
         // dprintf("Set sample_rate = %d \n",(long)AudioFormat.sample_rate);
         // dprintf("Set channels = %d \n",AudioFormat.channels);
         // dprintf("Set format = %d \n",AudioFormat.format);
         // dprintf("Set big_endian = %d \n",AudioFormat.big_endian);
         // dprintf("Set buffer_header = %d \n",AudioFormat.buffer_header);
         // dprintf("Set write_buffer_size = %d \n",AudioFormat.write_buffer_size);
         // dprintf("Set record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
         return B_OK;

    case B_AUDIO_GET_AUDIO_FORMAT: // arg = ptr to struct audio_format
         // dprintf("PPA: GET AUDIO FORMAT\n");
         // dprintf("Get sample_rate = %d \n",(long)AudioFormat.sample_rate);
         // dprintf("Get channels = %d \n",AudioFormat.channels);
         // dprintf("Get format = %d \n",AudioFormat.format);
         // dprintf("Get big_endian = %d \n",AudioFormat.big_endian);
         // dprintf("Get buffer_header = %d \n",AudioFormat.buffer_header);
         // dprintf("Get write_buffer_size = %d \n",AudioFormat.write_buffer_size);
         // dprintf("Get record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
         memcpy(data_buffer, &AudioFormat, sizeof(audio_format)); return B_OK;

    case B_AUDIO_GET_PREFERRED_SAMPLE_RATES: // arg = ptr to float[4]
         // dprintf("PPA: GET PREFERRED SAMPLE RATES\n");
         // dprintf("Rate 0 = %d \n",(long)rates[0]); // dprintf("Rate 1 = %d \n",(long)rates[1]);
         // dprintf("Rate 2 = %d \n",(long)rates[2]); // dprintf("Rate 3 = %d \n\n",(long)rates[3]);
         memcpy(data_buffer, rates, sizeof(rates)); return B_OK; 

    case SOUND_WRITE_BUFFER: // Play sound sample attached to audio_buffer_header.
         // dprintf("PPA: WRITE BUFFER\n");
         header = (audio_buffer_header *)data_buffer;        // Assign audio_buffer_header struct to data buffer.
         data_length = header->reserved_1 - sizeof(*header); // Sound data size after audio_buffer_header struct.
         if ( next_buffer_time < system_time() ) {           // First sample out or a long silence occurred.
            long sampling_rate, frame_length;
            sampling_rate = (long)(AudioFormat.sample_rate); // Samples per second to be sent to parallel port.
            frame_length = 1000000L / sampling_rate;         // How many (integer) microseconds between samples.
            next_buffer_time = (bigtime_t)((1+(system_time()/frame_length))*frame_length); } // Quantum Timing.
         header->time = next_buffer_time;                    // Time this buffer's data will start playing.
         header->sample_clock = next_buffer_time;            // Number of microseconds to play one second of sound?
         device_write(cookie, 0, header + 1, &data_length);  // Play sound data.
         return B_OK;                                        // Release semaphore to help debugging.
 
    case SOUND_READ_BUFFER:
         // dprintf("PPA: READ BUFFER\n");
         header = (audio_buffer_header *)data_buffer;       // Assign audio_buffer_header struct to data buffer.
         data_length = header->reserved_1 - sizeof(*header); // Sound data size after audio_buffer_header struct.
         header->time = system_time();                      // Time this buffer's data will start recording.
         header->sample_clock = system_time();              // Number of microseconds to play one second of sound?
         device_read(cookie, 0, header + 1, &data_length);  // Record sound data.
         release_sem(read_done_sem); return B_OK;           // Release semaphore to help debugging.

    case SOUND_SET_PLAYBACK_COMPLETION_SEM:
         // dprintf("PPA: SET PLAYBACK SEM\n");
         write_done_sem = *((sem_id *)data_buffer); return B_OK;

    case SOUND_SET_CAPTURE_COMPLETION_SEM:
         // dprintf("PPA: SET CAPTURE SEM\n\n");
         read_done_sem = *((sem_id *)data_buffer); return B_OK;

    case SOUND_SET_CAPTURE_PREFERRED_BUF_SIZE:
         // dprintf("PPA: SET CAPTURE SIZE\n");
         // dprintf("Old record_buffer_size = %d \n",AudioFormat.record_buffer_size);
         AudioFormat.record_buffer_size = (int32)data_buffer;
         // dprintf("Set record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
         return B_OK;

    case SOUND_SET_PLAYBACK_PREFERRED_BUF_SIZE:
         // dprintf("PPA: SET PLAYBACK SIZE\n");
         // dprintf("Old write_buffer_size = %d \n",AudioFormat.write_buffer_size);
         AudioFormat.write_buffer_size = (int32)data_buffer;
         // dprintf("Set write_buffer_size = %d \n\n",AudioFormat.write_buffer_size);
         return B_OK;

    case SOUND_GET_CAPTURE_PREFERRED_BUF_SIZE:
         // dprintf("PPA: GET CAPTURE SIZE\n");
         // dprintf("Get record_buffer_size = %d \n\n",AudioFormat.record_buffer_size);
         *(int32*)data_buffer = AudioFormat.record_buffer_size; return B_OK;

    case SOUND_GET_PLAYBACK_PREFERRED_BUF_SIZE:
         // dprintf("PPA: GET GET_PLAYBACK SIZE\n");
         // dprintf("Get write_buffer_size = %d \n\n",AudioFormat.write_buffer_size);
         *(int32*)data_buffer = AudioFormat.write_buffer_size; return B_OK;
 
    case B_ROUTING_GET_VALUES: // arg = ptr to struct audio_routing_cmd
         // dprintf("PPA: B_ROUTING_GET_VALUES Command.\n\n"); return B_ERROR;
    case B_ROUTING_SET_VALUES: // arg = ptr to struct audio_routing_cmd
         // dprintf("PPA: B_ROUTING_SET_VALUES Command.\n\n"); return B_ERROR;
    case B_MIXER_GET_VALUES: // arg = ptr to struct audio_routing_cmd
         // dprintf("PPA: B_MIXER_GET_VALUES Command.\n\n"); return B_ERROR;
    case B_MIXER_SET_VALUES: // arg = ptr to struct audio_routing_cmd
         // dprintf("PPA: B_MIXER_SET_VALUES Command.\n\n"); return B_ERROR;
    case B_AUDIO_GET_TIMING: // arg = ptr to struct audio_timing, used to be SV_SECRET_HANDSHAKE (10100)
         // dprintf("PPA: B_AUDIO_GET_TIMING Command.\n\n"); return B_ERROR;
    case SOUND_GET_PLAYBACK_TIMESTAMP:
         // dprintf("PPA: SOUND_GET_PLAYBACK_TIMESTAMP Command.\n\n"); return B_ERROR;
    case SOUND_GET_CAPTURE_TIMESTAMP:
         // dprintf("PPA: SOUND_GET_CAPTURE_TIMESTAMP Command.\n\n"); return B_ERROR;
    case SOUND_DEBUG_ON:
         // dprintf("PPA: SOUND_DEBUG_ON Command.\n\n"); return B_ERROR;
    case SOUND_DEBUG_OFF:
         // dprintf("PPA: SOUND_DEBUG_OFF Command.\n\n"); return B_ERROR;
    case SOUND_LOCK_FOR_DMA:
         // dprintf("PPA: SOUND_LOCK_FOR_DMA Command.\n\n"); return B_ERROR;

    // dprintf("PPA: Unknown Command Recieved=> %d \n\n",(int)opcode);
} return EINVAL; }

/****************************************************************************************************************/

status_t set_parameters(sound_setup *sound) {
   // dprintf("PPA: set_parameters.\n");
// Mixer sources to digitalize.
   left_adc_source = sound->left.adc_source; right_adc_source = sound->right.adc_source;
// Controls for channel.adc_gain 0..15 adc gain, in 1.5 dB steps.
   left_adc_gain = sound->left.adc_gain; right_adc_gain = sound->right.adc_gain;
// Set channel.mic levels, 1bit->2bit, non-zero enables 20 dB MIC input gain.
   left_mic_gain_enable  = sound->left.mic_gain_enable; right_mic_gain_enable = sound->right.mic_gain_enable;
// AUX-1 Settings - 0-31 aux1 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux1 mix.
   left_aux1_mix_gain  = sound->left.aux1_mix_gain;  left_aux1_mix_mute  = sound->left.aux1_mix_mute;
   right_aux1_mix_gain = sound->right.aux1_mix_gain; right_aux1_mix_mute = sound->right.aux1_mix_mute;
// AUX-2 Settings - 0-31 aux2 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux2 mix.
   left_aux2_mix_gain  = sound->left.aux2_mix_gain;  left_aux2_mix_mute  = sound->left.aux2_mix_mute;
   right_aux2_mix_gain = sound->right.aux2_mix_gain; right_aux2_mix_mute = sound->right.aux2_mix_mute;
// Line Levels Settings - 0-31 line mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes line mix.
   left_line_mix_gain  = sound->left.line_mix_gain;  left_line_mix_mute  = sound->left.line_mix_mute;
   right_line_mix_gain = sound->right.line_mix_gain; right_line_mix_mute = sound->right.line_mix_mute;
// DAC Settings - 0..61 dac attenuation, in -1.5 dB steps. Non-zero mutes dac output.
   left_dac_attn  = sound->left.dac_attn;  left_dac_mute  = sound->left.dac_mute;
   right_dac_attn = sound->right.dac_attn; right_dac_mute = sound->right.dac_mute;
// Only support 48 kHz. Sample rate from enum table.
   sound_sample_rate = sound->sample_rate;
// Only support 8 bit mono playback, sample format for playback.
   sound_playback_format = sound->playback_format;
// Only support 8 bit mono capture, sample format for capture.
   sound_capture_format = sound->capture_format;
// Control for dither_enable, non-zero enables dither on 16 => 8 bit.
   sound_dither_enable = sound->dither_enable;
// Control for loop_attn. 0..64 adc to dac loopback attenuation, in -1.5 dB steps. Non-zero enables loopback.
   sound_loop_attn = sound->loop_attn; sound_loop_enable = sound->loop_enable;
// Control for output boost, zero (2.0 Vpp) non-zero (2.8 Vpp) output level boost.
   sound_output_boost = sound->output_boost;
// Control for high pass filter, non-zero enables highpass filter in adc.
   sound_highpass_enable = sound->highpass_enable;
// Control for speaker gain, 0..64 mono speaker gain. Un/Mute speaker, non-zero mutes speaker.
   sound_mono_gain = sound->mono_gain; sound_mono_mute = sound->mono_mute;
/* // dprintf("PPA: ADC Sources (enum) => %d %d \n", (int)left_adc_source, (int)right_adc_source);
   // dprintf("PPA: ADC Gains => %d %d \n",   (int)left_adc_gain, (int)right_adc_gain);
   // dprintf("PPA: Mic Enables => %d %d \n", (int)left_mic_gain_enable, (int)right_mic_gain_enable);
   // dprintf("PPA: Left  Aux1 Gain-Mute => %d %d \n", (int)left_aux1_mix_gain,  (int)left_aux1_mix_mute);
   // dprintf("PPA: Right Aux1 Gain-Mute => %d %d \n", (int)right_aux1_mix_gain, (int)right_aux1_mix_mute);
   // dprintf("PPA: Left  Aux2 Gain-Mute => %d %d \n", (int)left_aux2_mix_gain,  (int)left_aux2_mix_mute);
   // dprintf("PPA: Right Aux2 Gain-Mute => %d %d \n", (int)right_aux2_mix_gain, (int)right_aux2_mix_mute);
   // dprintf("PPA: Left  Line Gain-Mute => %d %d \n", (int)left_line_mix_gain,  (int)left_line_mix_mute);
   // dprintf("PPA: Right Line Gain-Mute => %d %d \n", (int)right_line_mix_gain, (int)right_line_mix_mute);
   // dprintf("PPA: Left  DAC Mute-Attenuation => %d %d \n", (int)left_dac_mute,  (int)left_dac_attn);
   // dprintf("PPA: Right DAC Mute-Attenuation => %d %d \n", (int)right_dac_mute, (int)right_dac_attn);
   // dprintf("PPA: sound_sample_rate (enum) => %d \n", (int)sound_sample_rate);
   // dprintf("PPA: sound_playback_format (enum) => %d \n", (int)sound_playback_format);
   // dprintf("PPA: sound_capture_format (enum) => %d \n", (int)sound_capture_format);
   // dprintf("PPA: sound->dither_enable (enum) => %d \n", (int)sound_dither_enable);
   // dprintf("PPA: sound_loop_attn, sound_loop_enable => %d %d \n", (int)sound_loop_attn, (int)sound_loop_enable);
   // dprintf("PPA: sound->output_boost => %d \n", (int)sound_output_boost);
   // dprintf("PPA: sound_highpass_enable => %d \n", (int)sound_highpass_enable);
   // dprintf("PPA: Sound Mono Gain => %d  %d \n", (int)sound_mono_gain, (int)sound_mono_mute); */
   // dprintf("\n"); // Prints a gap for readability.
   return B_OK; }

/****************************************************************************************************************/
status_t get_parameters(sound_setup *sound) {

// Mixer sources to digitalize.
   sound->left.adc_source = left_adc_source; sound->right.adc_source = right_adc_source;
// Controls for channel.adc_gain. 0..15 adc gain, in 1.5 dB steps.
   sound->left.adc_gain = left_adc_gain; sound->right.adc_gain = right_adc_gain;
// Set channel.mic levels, 1bit->2bit, non-zero enables 20 dB MIC input gain.
   sound->left.mic_gain_enable  = left_mic_gain_enable; sound->right.mic_gain_enable = right_mic_gain_enable;
// AUX-1 Settings - 0-31 aux1 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux1 mix.
   sound->left.aux1_mix_gain  = left_aux1_mix_gain;  sound->left.aux1_mix_mute  = left_aux1_mix_mute;
   sound->right.aux1_mix_gain = right_aux1_mix_gain; sound->right.aux1_mix_mute = right_aux1_mix_mute;
// AUX-2 Settings - 0-31 aux2 mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes aux2 mix.
   sound->left.aux2_mix_gain  = left_aux2_mix_gain;  sound->left.aux2_mix_mute  = left_aux2_mix_mute;
   sound->right.aux2_mix_gain = right_aux2_mix_gain; sound->right.aux2_mix_mute = right_aux2_mix_mute;
// Line Levels Settings - 0-31 line mix to output gain. 12.0 to -34.5dB in 1.5dB steps. Non-zero mutes line mix.
   sound->left.line_mix_gain  = left_line_mix_gain;  sound->left.line_mix_mute  = left_line_mix_mute;
   sound->right.line_mix_gain = right_line_mix_gain; sound->right.line_mix_mute = right_line_mix_mute;
// DAC Settings - 0..61 dac attenuation, in -1.5 dB steps. Non-zero mutes dac output.
   sound->left.dac_attn  = left_dac_attn;  sound->left.dac_mute  = left_dac_mute;
   sound->right.dac_attn = right_dac_attn; sound->right.dac_mute = right_dac_mute;
// Only support 48 kHz. Sample rate from enum table.
   sound->sample_rate = sound_sample_rate;
// Only support 8 bit mono playback, sample format for playback.
   sound->playback_format = sound_playback_format;
// Only support 8 bit mono capture, sample format for capture.
   sound->capture_format = sound_capture_format;
// Control for dither_enable, non-zero enables dither on 16 => 8 bit.
   sound->dither_enable = sound->dither_enable;
// Control for loop_attn. 0..64 adc to dac loopback attenuation, in -1.5 dB steps. Non-zero enables loopback.
   sound->loop_attn = sound_loop_attn; sound->loop_enable = sound_loop_enable;
// Control for output boost, zero (2.0 Vpp) non-zero (2.8 Vpp) output level boost.
   sound->output_boost = sound_output_boost;
// Control for high pass filter, non-zero enables highpass filter in adc.
   sound->highpass_enable = sound_highpass_enable;
// Control for speaker gain, 0..64 mono speaker gain.
   sound->mono_gain = sound_mono_gain;
// Un/Mute speaker, non-zero mutes speaker.
   sound->mono_mute = sound_mono_mute;
   // dprintf("PPA: ADC Sources (enum) => %d %d \n", (int)left_adc_source, (int)right_adc_source);
   // dprintf("PPA: ADC Gains => %d %d \n",   (int)left_adc_gain, (int)right_adc_gain);
   // dprintf("PPA: Mic Enables => %d %d \n", (int)left_mic_gain_enable, (int)right_mic_gain_enable);
   // dprintf("PPA: Left  Aux1 Gain-Mute => %d %d \n", (int)left_aux1_mix_gain,  (int)left_aux1_mix_mute);
   // dprintf("PPA: Right Aux1 Gain-Mute => %d %d \n", (int)right_aux1_mix_gain, (int)right_aux1_mix_mute);
   // dprintf("PPA: Left  Aux2 Gain-Mute => %d %d \n", (int)left_aux2_mix_gain,  (int)left_aux2_mix_mute);
   // dprintf("PPA: Right Aux2 Gain-Mute => %d %d \n", (int)right_aux2_mix_gain, (int)right_aux2_mix_mute);
   // dprintf("PPA: Left  Line Gain-Mute => %d %d \n", (int)left_line_mix_gain,  (int)left_line_mix_mute);
   // dprintf("PPA: Right Line Gain-Mute => %d %d \n", (int)right_line_mix_gain, (int)right_line_mix_mute);
   // dprintf("PPA: Left  DAC Mute-Attenuation => %d %d \n", (int)left_dac_mute,  (int)left_dac_attn);
   // dprintf("PPA: Right DAC Mute-Attenuation => %d %d \n", (int)right_dac_mute, (int)right_dac_attn);
   // dprintf("PPA: sound_sample_rate (enum) => %d \n", (int)sound_sample_rate);
   // dprintf("PPA: sound_playback_format (enum) => %d \n", (int)sound_playback_format);
   // dprintf("PPA: sound_capture_format (enum) => %d \n", (int)sound_capture_format);
   // dprintf("PPA: sound->dither_enable (enum) => %d \n", (int)sound_dither_enable);
   // dprintf("PPA: sound_loop_attn, sound_loop_enable => %d %d \n", (int)sound_loop_attn, (int)sound_loop_enable);
   // dprintf("PPA: sound->output_boost => %d \n", (int)sound_output_boost);
   // dprintf("PPA: sound_highpass_enable => %d \n", (int)sound_highpass_enable);
   // dprintf("PPA: Sound Mono Gain => %d  %d \n\n", (int)sound_mono_gain, (int)sound_mono_mute); */
   // dprintf("\n"); // Prints a gap for readability.
   return B_OK; }

/*****************************************************************************************************************
 read_hook() - status_t read_hook(void *cookie, off_t position, void *data, size_t *length) - This hook handles 
 the Posix read() function for an open instance of your driver.  Implement it to read len bytes of data starting
 at the specified byte position on the device, storing the read bytes at data. Exactly what this does is
 device-specific (disk devices would read from the specified offset on the disk, but a graphics driver might have
 some other interpretation of this request). Before returning, you should set len to the actual number of bytes
 read into the buffer. Return B_OK if data was read (even if the number of returned bytes is less than requested),
 otherwise return an appropriate error. */

static status_t device_read(void *cookie, off_t pos, void *data_buffer, size_t *length) {
 int index; long record_buffer_length;

 acquire_sem(read_block); // Block extra reads from buffer before it is loaded into system.
 index = 0; record_buffer_length = *length; // Setup to copy to data buffer.
 memcpy(data_buffer, record_buffer, record_buffer_length); // Copy fake sound data to play buffer.
                                                           // Fill read buffer with a test (sawtooth) wave form.
 release_sem_etc(read_done_sem, 1, B_DO_NOT_RESCHEDULE);   // Release semaphore after data copied to databuffer.
 release_sem(read_block); return B_OK; }                   // Allow more reads.

/*****************************************************************************************************************
 write_hook() - status_t write_hook(void *cookie, off_t position, void *data, size_t length) -  This hook handles
 the Posix write() function for an open instance of your driver. Implement it to write len bytes of data starting
 at the specified byte position on the device, from the buffer pointed to by data. Exactly what this does is
 device-specific (disk devices would write to the specified offset on the disk, but a graphics driver might have
 some other interpretation of this request). Return B_OK if data was read (even if the number of returned bytes is
 less than requested), otherwise return an appropriate error. */ 

static status_t device_write(void *cookie, off_t pos, const void *data_buffer, size_t *length) {
 int index; long data_buffer_length, play_buffer_length, sound_data_length, left_channel, rigth_channel;
 long sampling_rate, frame_length, frame_rate; bigtime_t next_frame_time, present_time, latency;

 acquire_sem(write_block); // Block additional writes to sound data buffer before it is played out.

 index = 0; data_buffer_length = (long)(*length); // Setup length to copy from data buffer.

 memcpy(play_buffer, data_buffer, data_buffer_length); // Copy sound data to play buffer.
 play_buffer_length = data_buffer_length / 2; // Number of 16 bit samples in play_buffer.

 release_sem_etc(write_done_sem, 1, B_DO_NOT_RESCHEDULE); // Release semaphore after data copied to play buffer.

 sound_data_length = play_buffer_length / 2; // Setup to copy from play buffer to sound buffer.
 for ( index = 0; index < play_buffer_length; index = index + 2 ) { // Create 8 bit version of sound data.
     left_channel  = 32768L + (long)play_buffer[index];     // Convert signed to unsigned.
     rigth_channel = 32768L + (long)play_buffer[index + 1]; // Convert signed to unsigned.
     // sound_data8[index/2] = (uint8)((left_channel + rigth_channel) / 512); } // Worse sound! Why?
     sound_data8[index/2] = (uint8)(left_channel / 256); }
 
 sampling_rate = (long)(AudioFormat.sample_rate); // Number of samples per second to be outputted to parallel port.
 frame_length = 1000000L / sampling_rate; // How many (integer) microseconds between samples.
 frame_rate   = 1000000L / frame_length;  // The rate the OS's interrupts allows samples to be played at.
 next_buffer_time = next_buffer_time + (bigtime_t)(sound_data_length * frame_length); // End of present buffer.
 next_frame_time = (bigtime_t)(1 + (system_time() / frame_length)) * frame_length; // Quantum Timing of sound out.
 
 if (info_write) { // Print out info to insure frequency changes are correct.
    // dprintf("Sampling Rate %d Frame Length %d Frame Rate %d\n", sampling_rate, frame_length, frame_rate);
    // dprintf("Data Size = %d \n",play_buffer_length);
    info_write = false; }
 for (index = 0; index < sound_data_length; index++ ) {
     latency = (bigtime_t)(Latency_Timing / 100); // Latency in microseconds. 
     snooze_until( next_frame_time - latency, B_SYSTEM_TIMEBASE); // Wait until just before time to output.
     present_time = system_time(); // Find what time was called to output next sample.
     isa->write_io_8(0x378, sound_data8[index]); // Write data out to parallel port.
     if ( present_time > next_frame_time ) { Latency_Timing++; } // Snooze lasted too long.
     if ( present_time < next_frame_time ) { Latency_Timing--; } // Snooze ended too early.
     next_frame_time = next_frame_time + frame_length; }  // Calculate time of next output.
 release_sem(write_block); return B_OK; } // Allow new calls to device_write.

/*****************************************************************************************************************
 publish_devices - const char** publish_devices(void) - returns a null-terminated array of devices supported by
 this driver.  Devfs calls publish_devices() to learn the names, relative to /dev, of the devices the driver
 supports. The driver should return a NULL-terminated array of strings indicating all the installed devices the
 driver supports. For example, an ethernet device driver might return: 
 static char *devices[] = { "net/ether", NULL };

 In this case, devfs will then create the pseudo-file /dev/net/ether, through which all user applications can
 access the driver.  Since only one instance of the driver will be loaded, if support for multiple devices of the
 same type is desired, the driver must be capable of supporting them. If the driver senses (and supports) two
 ethernet cards, it might return: 
 static char *devices[] = { "net/ether1", "net/ether2", NULL }; */

const char** publish_devices(void) {
 return driver_names; }

/*****************************************************************************************************************
 find_device - device_hooks* find_device(const char *name) - returns a pointer to device hooks structure for a
 given device name.  When a device published by the driver is accessed, devfs communicates with it through a
 series of hook functions that handle the requests.  The find_device() function is called to obtain a list of
 these hook functions, so that devfs can call them.  The device_hooks structure returned lists out the hook
 functions.  The device_hooks structure, and what each hook does, is described in the next section. */

device_hooks *find_device(const char *name) {
 int i; // dprintf("PPA: Look at device names.\n"); 
 for (i=0; driver_names[i]; i++) {
     if (!strncmp(name, driver_names[i], B_OS_NAME_LENGTH)) {
        // dprintf("PPA: Found matching %s driver.\n\n",driver_names[i]);
        return &sound_device[i]; } }
 // dprintf("PPA: No driver match found.\n\n");
 return NULL; }

/*****************************************************************************************************************
 status_t readv_hook(void *cookie, off_t position, const struct iovec *vec, size_t count, size_t *length)
 This hook handles the Posix readv() function for an open instance of your driver.  This is a scatter/gather read
 function; given an array of iovec structures describing address/length pairs for a group of destination buffers,
 your implementation should fill each successive buffer with bytes, up to a total of len bytes. The vec array has
 count items in it. As with read_hook(), set len to the actual number of bytes read, and return an appropriate
 result code.

static status_t my_device_readv(void *cookie, off_t position, const iovec *vec, size_t count, size_t *length) {
 // dprintf("PPA: Readv driver.\n");
 return B_OK; }

/*****************************************************************************************************************
 status_t writev_hook(void *cookie, off_t position, const struct iovec *vec, size_t count, size_t *length) 
 This hook handles the Posix writev() function for an open instance of your driver. This is a scatter/gather write
 function; given an array of iovec structures describing address/length pairs for a group of source buffers, your
 implementation should write each successive buffer to disk, up to a total of len bytes. The vec array has count
 items in it. Before returning, set len to the actual number of bytes written, and return an appropriate result
 code.

static status_t my_device_writev(void *cookie, off_t position, const iovec *vec, size_t count, size_t *length) {
 // dprintf("PPA: Writev driver.\n");
 return B_OK; }

/*****************************************************************************************************************
select_hook() , deselect_hook() 
 These hooks are reserved for future use. Set the corresponding entries in your device_hooks structure to NULL.

status_t my_device_select(void *cookie, uint8 event, uint32 ref, selectsync *sync) {
 // dprintf("PPA: Select driver.\n");
 return B_OK; }

status_t my_device_deselect(void *cookie, uint8 event, selectsync *sync) {
 // dprintf("PPA: Deselect driver.\n");
 return B_OK; } */
